<?php if ( ! defined( 'ABSPATH' ) ) exit;

if ( ! class_exists( 'NF_Abstracts_PaymentGateway' ) ) return;

/**
 * The Payeezy payment gateway for the Collect Payment action.
 */
class NF_Payeezy_PaymentGateway extends NF_Abstracts_PaymentGateway {

    protected $_slug = 'payeezy';

    protected $form_ids = array();

    protected $api_key;
    protected $api_secret;
    protected $merchant_token;
    protected $test_mode;

    public function __construct() {
        parent::__construct();

        $this->_name = __( 'Payeezy', 'nf-payeezy' );

		$this->_live_url = 'https://api.payeezy.com/v1/transactions';
		$this->_test_url = 'https://api-cert.payeezy.com/v1/transactions';

        add_action( 'init', array( $this, 'init' ) );

		$this->_settings[ 'payeezy_firstname' ] = array(
            'name' => 'payeezy_firstname',
            'type' => 'textbox',
            'label' => __( 'First Name', 'nf-payeezy' ),
            'width' => 'full',
            'group' => 'primary',
            'deps'  => array(
                'payment_gateways' => $this->_slug,
            ),
            'help' => __( 'Leave this field blank if you want the plugin to automatically pick the value from the form.', 'nf-payeezy' ),
            'use_merge_tags' => TRUE
        );

		$this->_settings[ 'payeezy_lastname' ] = array(
            'name' => 'payeezy_lastname',
            'type' => 'textbox',
            'label' => __( 'Last Name', 'nf-payeezy' ),
            'width' => 'full',
            'group' => 'primary',
            'deps'  => array(
                'payment_gateways' => $this->_slug,
            ),
            'help' => __( 'Leave this field blank if you want the plugin to automatically pick the value from the form.', 'nf-payeezy' ),
            'use_merge_tags' => TRUE
        );

        $this->_settings[ 'payeezy_address' ] = array(
            'name' => 'payeezy_address',
            'type' => 'textbox',
            'label' => __( 'Address', 'nf-payeezy' ),
            'width' => 'full',
            'group' => 'primary',
            'deps'  => array(
                'payment_gateways' => $this->_slug,
            ),
            'help' => __( 'Leave this field blank if you want the plugin to automatically pick the value from the form.', 'nf-payeezy' ),
            'use_merge_tags' => TRUE
        );

        $this->_settings[ 'payeezy_city' ] = array(
            'name' => 'payeezy_city',
            'type' => 'textbox',
            'label' => __( 'City', 'nf-payeezy' ),
            'width' => 'full',
            'group' => 'primary',
            'deps'  => array(
                'payment_gateways' => $this->_slug,
            ),
            'help' => __( 'Leave this field blank if you want the plugin to automatically pick the value from the form.', 'nf-payeezy' ),
            'use_merge_tags' => TRUE
        );

        $this->_settings[ 'payeezy_state' ] = array(
            'name' => 'payeezy_state',
            'type' => 'textbox',
            'label' => __( 'State', 'nf-payeezy' ),
            'width' => 'full',
            'group' => 'primary',
            'deps'  => array(
                'payment_gateways' => $this->_slug,
            ),
            'help' => __( 'Leave this field blank if you want the plugin to automatically pick the value from the form.', 'nf-payeezy' ),
            'use_merge_tags' => TRUE
        );

        $this->_settings[ 'payeezy_country' ] = array(
            'name' => 'payeezy_country',
            'type' => 'textbox',
            'label' => __( 'Country', 'nf-payeezy' ),
            'width' => 'full',
            'group' => 'primary',
            'deps'  => array(
                'payment_gateways' => $this->_slug,
            ),
            'help' => __( 'Leave this field blank if you want the plugin to automatically pick the value from the form.', 'nf-payeezy' ),
            'use_merge_tags' => TRUE
        );

        $this->_settings[ 'payeezy_zip' ] = array(
            'name' => 'payeezy_zip',
            'type' => 'textbox',
            'label' => __( 'Zip/Postal Code', 'nf-payeezy' ),
            'width' => 'full',
            'group' => 'primary',
            'deps'  => array(
                'payment_gateways' => $this->_slug,
            ),
            'help' => __( 'Leave this field blank if you want the plugin to automatically pick the value from the form.', 'nf-payeezy' ),
            'use_merge_tags' => TRUE
        );

        $this->_settings[ 'payeezy_phone' ] = array(
            'name' => 'payeezy_phone',
            'type' => 'textbox',
            'label' => __( 'Phone', 'nf-payeezy' ),
            'width' => 'full',
            'group' => 'primary',
            'deps'  => array(
                'payment_gateways' => $this->_slug,
            ),
            'help' => __( 'Leave this field blank if you want the plugin to automatically pick the value from the form.', 'nf-payeezy' ),
            'use_merge_tags' => TRUE
        );

		$this->_settings[ 'payeezy_email' ] = array(
            'name' => 'payeezy_email',
            'type' => 'textbox',
            'label' => __( 'Email', 'nf-payeezy' ),
            'width' => 'full',
            'group' => 'primary',
            'deps'  => array(
                'payment_gateways' => $this->_slug,
            ),
            'help' => __( 'Leave this field blank if you want the plugin to automatically pick the value from the form.', 'nf-payeezy' ),
            'use_merge_tags' => TRUE
        );

		$this->_settings[ 'payeezy_merchant_ref' ] = array(
            'name' => 'payeezy_merchant_ref',
            'type' => 'textbox',
            'label' => __( 'Merchant Reference', 'nf-payeezy' ),
            'width' => 'full',
            'group' => 'primary',
            'deps'  => array(
                'payment_gateways' => $this->_slug,
            ),
            'help' => __( 'It is used by Payeezy system will be reflected in your settlement records.', 'nf-payeezy' ),
            'use_merge_tags' => TRUE
        );
    }

	public function init() {
        $this->api_key = Ninja_Forms()->get_setting( 'payeezy_api_key' );
        $this->api_secret = Ninja_Forms()->get_setting( 'payeezy_api_secret' );
        $this->merchant_token = Ninja_Forms()->get_setting( 'payeezy_merchant_token' );
        $this->test_mode = Ninja_Forms()->get_setting( 'payeezy_test_mode' );
    }

	private function is_payeezy_active( $form_id ) {
        $form_actions = Ninja_Forms()->form( $form_id )->get_actions();

        foreach( $form_actions as $action ) {
            if( 'collectpayment' != $action->get_setting( 'type' ) ) continue;
            if( $this->_slug != $action->get_setting( 'payment_gateways' ) ) continue;

            return true;
        }
        return false;
    }

    /**
     * Process
     *
     * The main function for processing submission data.
     *
     * @param array $action_settings Action specific settings.
     * @param int $form_id The ID of the submitted form.
     * @param array $data Form submission data.
     * @return array $data Modified submission data.
     */
    public function process( $action_settings, $form_id, $data ) {

        $total = (int) round( $action_settings[ 'payment_total' ] * 100 ); //Processed Amount in cents. 1299 = $12.99

        if ( ! $total ) {
            return $data;
        }

		if ( ! $this->api_key || ! $this->api_secret || ! $this->merchant_token ) {
			$data[ 'errors' ][ 'form' ][ 'payeezy' ] = __( 'Missing Payeezy (First Data) credentials.', 'nf-payeezy' );
            return $data;
		}

        // Load the payeezy libraries
        if ( ! class_exists( 'Payeezy' ) ) {
            require_once( NF_Payeezy::$dir . 'api/Payeezy.php' );
        }

		$is_sandbox = $this->test_mode == true;
		$api_endpoint = $is_sandbox ? $this->_test_url : $this->_live_url;

		$transaction = new Payeezy();

		$transaction::setApiKey( $this->api_key );
    	$transaction::setApiSecret( $this->api_secret );
    	$transaction::setMerchantToken( $this->merchant_token );
        $transaction::setUrl( $api_endpoint );

		$payment_args = $this->get_payment_args( $total, $data, $action_settings );

		$response = json_decode( $transaction->purchase( $payment_args ) );

		if ( isset( $response->validation_status ) && 'success' === $response->validation_status && 'approved' === $response->transaction_status ) {

			$this->update_submission( $this->get_sub_id( $data ), array(
				'payeezy_charge_id' => $response->transaction_id,
				'payeezy_last4' => substr( $this->get_form_value_by_type( 'creditcardnumber', $data ), -4 ),
				'payeezy_brand' => $response->card->type,
				'payeezy_live' => ! $this->test_mode,
			));

		}  else {

			$error_message = $this->extract_payment_error( $response );
			$data[ 'errors' ][ 'form' ][ 'payeezy' ] = $error_message;

		}

        return $data;
    }

	public function get_payment_args( $total, $data, $action_settings ) {
		$payment_args = array();

		$payment_args['amount'] = $total;

		$card_number = $this->get_form_value_by_type( 'creditcardnumber', $data );

		$payment_args['card_type'] = $this->get_card_type( $card_number, 'pattern', 'label' );
		$payment_args['card_number'] = $card_number;

		$exp_month = substr( $this->get_form_value_by_type( 'creditcardexpiration', $data ), 0, 2 );
		$exp_year = substr( $this->get_form_value_by_type( 'creditcardexpiration', $data ), -2 );

		$payment_args['card_expiry'] = $exp_month . $exp_year;
		$payment_args['card_cvv'] = $this->get_form_value_by_type( 'creditcardcvc', $data );

		$payment_args['card_holder_name'] = $this->get_form_value_by_type( 'creditcardfullname', $data );
		$payment_args['currency_code'] = $this->get_currency( $data );
		$payment_args['method'] = 'credit_card';
		$payment_args['merchant_ref'] = $action_settings[ 'payeezy_merchant_ref' ] ? $action_settings[ 'payeezy_merchant_ref' ] : $action_settings[ 'label' ];

		$card_name  = $this->get_first_last_name( $this->get_form_value_by_type( 'creditcardfullname', $data ) );
		$first_name = $action_settings[ 'payeezy_firstname' ] ? $action_settings[ 'payeezy_firstname' ] : ( $this->get_form_value_by_type( 'firstname', $data )  ? $this->get_form_value_by_type( 'firstname', $data ) : $card_name['firstname']);
		$last_name	= $action_settings[ 'payeezy_lastname' ] ? $action_settings[ 'payeezy_lastname' ] : ( $this->get_form_value_by_type( 'lastname', $data )  ? $this->get_form_value_by_type( 'lastname', $data ) : $card_name['lastname']);
		$address 	= $action_settings[ 'payeezy_address' ] ? $action_settings[ 'payeezy_address' ] : $this->get_form_value_by_type( 'address', $data );
        $city 		= $action_settings[ 'payeezy_city' ] ? $action_settings[ 'payeezy_city' ] : $this->get_form_value_by_type( 'city', $data );
        $state 		= $action_settings[ 'payeezy_state' ] ? $action_settings[ 'payeezy_state' ] : $this->get_form_value_by_type( 'liststate', $data );
        $country 	= $action_settings[ 'payeezy_country' ] ? $action_settings[ 'payeezy_country' ] : $this->get_form_value_by_type( 'listcountry', $data );
        $zip 		= $action_settings[ 'payeezy_zip' ] ? $action_settings[ 'payeezy_zip' ] : ( $this->get_form_field_by_type( 'creditcardzip', $data ) ? $this->get_form_value_by_type( 'creditcardzip', $data ) : $this->get_form_value_by_type( 'zip', $data ) );
        $phone 		= $action_settings[ 'payeezy_phone' ] ? $action_settings[ 'payeezy_phone' ] : $this->get_form_value_by_type( 'phone', $data );
		$email 		= $action_settings[ 'payeezy_email' ] ? $action_settings[ 'payeezy_email' ] : $this->get_form_value_by_type( 'email', $data );

		$payment_args['name'] = $first_name . ' ' . $last_name;
        $payment_args['city'] = $city;
        $payment_args['country'] = $country;
		$payment_args['email'] = $email;
		$payment_args['street'] = $address;
		$payment_args['state_province'] = $state;
		$payment_args['zip_postal_code'] = $zip;
		$payment_args['type'] = 'Work';
		$payment_args['number'] = $phone;

		return $payment_args;
	}

	function get_card_type( $value, $field = 'pattern', $return = 'label' ) {
		$card_types = array(
			array(
				'label' => 'American Express',
				'name' => 'amex',
				'pattern' => '/^3[47]/',
				'valid_length' => '[15]'
			),
			array(
				'label' => 'JCB',
				'name' => 'jcb',
				'pattern' => '/^35(2[89]|[3-8][0-9])/',
				'valid_length' => '[16]'
			),
			array(
				'label' => 'Discover',
				'name' => 'discover',
				'pattern' => '/^(6011|622(12[6-9]|1[3-9][0-9]|[2-8][0-9]{2}|9[0-1][0-9]|92[0-5]|64[4-9])|65)/',
				'valid_length' => '[16]'
			),
			array(
				'label' => 'MasterCard',
				'name' => 'mastercard',
				'pattern' => '/^5[1-5]/',
				'valid_length' => '[16]'
			),
			array(
				'label' => 'Visa',
				'name' => 'visa',
				'pattern' => '/^4/',
				'valid_length' => '[16]'
			),
			array(
				'label' => 'Maestro',
				'name' => 'maestro',
				'pattern' => '/^(5018|5020|5038|6304|6759|676[1-3])/',
				'valid_length' => '[12, 13, 14, 15, 16, 17, 18, 19]'
			),
			array(
				'label' => 'Diners Club',
				'name' => 'diners-club',
				'pattern' => '/^3[0689]/',
				'valid_length' => '[14]'
			),
		);

		foreach( $card_types as $type ) {
			$card_type = $type['name'];
			$compare = $type[$field];
			if ( ( $field == 'pattern' && preg_match( $compare, $value, $match ) ) || $compare == $value ) {
				return $type[$return];
			}
		}

	}

	public function extract_payment_error( $response ) {
        if ( ( isset( $response->validation_status ) && 'failed' === $response->validation_status ) || ( isset( $response->status ) && 'failed' === $response->status ) ) {
			$error_message = $response->Error->messages[0]->description;
		} elseif ( isset( $response->validation_status ) && 'success' === $response->validation_status && isset( $response->transaction_status ) && 'approved' !== $response->transaction_status ) {
            $error_message = __( 'Your card has been declined.', 'nf-payeezy' );
		} else {
			$error_message = __( 'There was a problem processing your transaction.', 'nf-payeezy' );
		}
        return $error_message;
    }

    private function get_form_field_by_type( $field_type, $data ) {
        foreach( $data[ 'fields' ] as $field ) {
            if( $field_type == $field[ 'type' ] ) return $field[ 'id' ];
        }
        return false;
    }

	private function get_form_value_by_type( $field_type, $data, $index = 1 ) {
        $match = 1;
        foreach( $data[ 'fields' ] as $field ) {
            if( $field_type == $field[ 'type' ] ) {
                if( $match == $index ) {
                    return $field[ 'value' ];
                } else {
                    $match++;
                }
            }
        }
        return false;
    }

    private function update_submission( $sub_id, $data = array() ) {
        if( ! $sub_id ) return;

        $sub = Ninja_Forms()->form()->sub( $sub_id )->get();

        foreach( $data as $key => $value ) {
            $sub->update_extra_value( $key, $value );
        }

        $sub->save();
    }

    private function get_sub_id( $data ) {
        if( isset( $data[ 'actions' ][ 'save' ][ 'sub_id' ] ) ) {
            return $data[ 'actions' ][ 'save' ][ 'sub_id' ];
        }
        return FALSE;
    }

    private function get_first_last_name( $text ) {
		$names      = explode( ' ', $text );
		$first_name = $names ? $names[0] : '';
		$last_name  = '';
		if ( count( $names ) > 1 ) {
			$last_name = $names[count( $names ) - 1];
		}

		$names_array = array( 'firstname' => $first_name, 'lastname' => $last_name );

		return $names_array;
	}

	private function get_currency( $form_data ) {
        /**
         * Currency Setting Priority
         *
         * 2. Ninja Forms Currency Setting
         * 1. Form Currency Setting (default)
         */
        $plugin_currency = Ninja_Forms()->get_setting( 'currency', 'USD' );
        $form_currency = ( isset( $form_data[ 'settings' ][ 'currency' ] ) && $form_data[ 'settings' ][ 'currency' ] ) ? $form_data[ 'settings' ][ 'currency' ] : $plugin_currency;
        return $form_currency;
    }

} // END CLASS NF_Payeezy_PaymentGateway
