<?php
namespace Deployer;

use React\Stream\Util;

require dirname( __DIR__).'/src/utils.php';

set('new_password', generate_random_string(32));
set('dev_site', defined('SITE') ? substr(SITE, 0, strrpos(SITE, '.')).'.dojocreative.net' : 'staging.example.com');
set('wordpress_content_dir', get('wordpress_content_dir', 'public/wp-content'));

desc('Check for existing deploy folder');
task('initial-deploy:check-domain-folder', function () {
    if (test('[ -d {{deploy_path}} ]')) {
        writeln('<comment>Deploy path already exists on the server: {{deploy_path}}</comment>');
        if (!askConfirmation('It is not wise to continue. Continue anyway?', false)) {
            throw new \RuntimeException('Initial deploy aborted by user.');
        }
    }
});

desc('Ensure remote SSH key exists');
task('initial-deploy:ensure-ssh-key', function () {
    $pubKeyPath = '~/.ssh/id_ed25519.pub';
    $hasKey = test('[ -f ~/.ssh/id_ed25519.pub ]');

    if (!$hasKey) {
        $comment = defined('SITE') ? 'deploy@'.SITE : 'deploy@remote';
        run('ssh-keygen -t ed25519 -f ~/.ssh/id_ed25519 -N "" -C '.$comment);
    }

    if (!test('[ -f ~/.ssh/id_ed25519.pub ]')) {
        throw new \RuntimeException('No SSH public key found on remote after generation.');
    }

    set('ssh_pubkey_path', $pubKeyPath);
    writeln('<info>Using remote SSH public key: '.$pubKeyPath.'</info>');
});

desc('Add deploy key to GitLab');
task('initial-deploy:gitlab-deploy-key', function () {
    $pubKeyPath = get('ssh_pubkey_path');
    if (!test('[ -f '.$pubKeyPath.' ]')) {
        throw new \RuntimeException('SSH public key not found on remote at '.$pubKeyPath);
    }

    $gitlabHost = rtrim(ask('GitLab base URL?', 'https://gitlab.com'), '/');
    $project = ask('GitLab project ID or URL-encoded path (e.g. group%2Fproject)');
    $token = askHidden('GitLab personal access token (api scope)');

    $title = 'dojo-deploy-'.date('Ymd-His');
    $pubKey = trim(run('cat '.$pubKeyPath));

    $listCmd = sprintf(
        'curl --fail --silent --show-error --header %s %s',
        escapeshellarg('PRIVATE-TOKEN: '.$token),
        escapeshellarg($gitlabHost.'/api/v4/projects/'.$project.'/deploy_keys')
    );

    $existingKeys = runLocally($listCmd);
    if (is_string($existingKeys) && strpos($existingKeys, $pubKey) !== false) {
        writeln('<info>GitLab deploy key already exists. Skipping add.</info>');
        return;
    }

    $cmd = sprintf(
        'curl --fail --silent --show-error --request POST --header %s --data-urlencode %s --data-urlencode %s --data-urlencode %s %s',
        escapeshellarg('PRIVATE-TOKEN: '.$token),
        escapeshellarg('title='.$title),
        escapeshellarg('key='.$pubKey),
        escapeshellarg('can_push=false'),
        escapeshellarg($gitlabHost.'/api/v4/projects/'.$project.'/deploy_keys')
    );

    runLocally($cmd);
});

desc('Creating Database and Users');
task('initial-deploy:create-database', function () {

    $server = currentHost();
    $db_name = $server->getRemoteUser().'_live';
    $db_user = $server->getRemoteUser().'_wp';

    run('uapi Mysql create_database name='.$db_name);
    run('uapi Mysql create_user name='.$db_user. ' password={new_password}');
    run('uapi Mysql set_privileges_on_database user='.$db_user.' database='.$db_name.' privileges=ALL');
});


desc('Creating .env file');
task('initial-deploy:create-env', function () {

    $server = currentHost();
    $testDomain = ask('What is the test domain?', get('dev_site'));
    $testDomain = preg_replace('#^https?://#', '', $testDomain);
    set('test_domain', $testDomain);
    set('wp_home', 'https://'.$testDomain);

    // Set our production values
    $info['WP_ENVIRONMENT_TYPE'] = 'development';
    $info['DB_NAME'] = $server->getRemoteUser().'_live';
    $info['DB_USER'] = $server->getRemoteUser().'_wp';
    $info['DB_PASSWORD'] = get('new_password');
    $info['WP_HOME'] = get('wp_home');

    /// Set our Keys
    $info['AUTH_KEY'] = $info['AUTH_KEY'] ?? generate_random_string(65, "'");
    $info['SECURE_AUTH_KEY'] = $info['SECURE_AUTH_KEY'] ?? generate_random_string(65, "'");
    $info['LOGGED_IN_KEY'] = $info['LOGGED_IN_KEY'] ?? generate_random_string(65, "'");
    $info['NONCE_KEY'] = $info['NONCE_KEY'] ?? generate_random_string(65, "'");
    $info['AUTH_SALT'] = $info['AUTH_SALT'] ?? generate_random_string(65, "'");
    $info['SECURE_AUTH_SALT'] = $info['SECURE_AUTH_SALT'] ?? generate_random_string(65, "'");
    $info['LOGGED_IN_SALT'] = $info['LOGGED_IN_SALT'] ?? generate_random_string(65, "'");
    $info['NONCE_SALT'] = $info['NONCE_SALT'] ?? generate_random_string(65, "'");

    cd('{{deploy_path}}/shared');
    run('echo "# Staging Config" > .env');

    foreach ($info as $key => $value) {
        run('echo "'.$key.'='.$value.'" >> .env');
    }

    cd('{{current_path}}');
});

desc('Export local database');
task('initial-deploy:db-export-local', function () {
    runLocally('mkdir -p .dep');
    runLocally('wp db export .dep/local.sql');
});

desc('Upload local database');
task('initial-deploy:db-upload', function () {
    run('mkdir -p {{deploy_path}}/.dep');
    upload('.dep/local.sql', '{{deploy_path}}/.dep/local.sql');
});

desc('Import database on remote');
task('initial-deploy:db-import-remote', function () {
    cd('{{current_path}}');
    run('wp db import {{deploy_path}}/.dep/local.sql');
    run('rm {{deploy_path}}/.dep/local.sql');
});

desc('Search and replace domain');
task('initial-deploy:search-replace', function () {
    $oldUrl = ask('What is the original test site URL?');
    $newUrl = get('wp_home');

    cd('{{current_path}}');
    run('wp elementor replace_urls '.$oldUrl.' '.$newUrl);
    run('wp search-replace '.$oldUrl.' '.$newUrl.' --skip-columns=guid --all-tables');
    run('wp cache flush');
});

desc('Sync uploads folder');
task('initial-deploy:sync-uploads', function () {
    $server = currentHost();
    $host = $server->getHostname();
    $port = $server->getPort() ? ' -p' . $server->getPort() : '';
    $sshArguments = $server->getSshArguments();
    $user = !$server->getRemoteUser() ? '' : $server->getRemoteUser() . '@';

    runLocally("rsync -vrz -e 'ssh$port $sshArguments' '{{wordpress_content_dir}}/uploads/' '$user$host:{{deploy_path}}/current/{{wordpress_content_dir}}/uploads/'", ['tty' => true]);
});

desc('Cleanup local temp files');
task('initial-deploy:cleanup-local', function () {
    runLocally('rm -f .dep/local.sql');
});

desc('Initial Deploy');
task('initial-deploy', [
    'initial-deploy:check-domain-folder',
    'initial-deploy:ensure-ssh-key',
    'initial-deploy:gitlab-deploy-key',
    'deploy',
    'initial-deploy:create-database',
    'initial-deploy:create-env',
    'initial-deploy:db-export-local',
    'initial-deploy:db-upload',
    'initial-deploy:db-import-remote',
    'initial-deploy:search-replace',
    'initial-deploy:sync-uploads',
    'initial-deploy:cleanup-local',
]);
