<?php
/**
 * Plugin Name: Dojo Category Selector Widget
 * Description: A custom Wordpress widget that displays categories for (custom) post types.
 * Version: 1
 * Author: Ryan Bronkema for Dojo Creative
 * Author URI: http://letsgodojo.com
 * Tags: custom taxonomy, custom tax, widget, sidebar, category, categories, taxonomy, custom category, custom categories, post types, custom post types, custom post type categories
 * License: GPL
 */

// Register 'Category Selector' widget
add_action( 'widgets_init', 'init_dojo_cs' );
function init_dojo_cs() { return register_widget('dojo_cs'); }

class dojo_cs extends WP_Widget {
    /** constructor */
    function __construct() {
        parent::__construct( 'dojo_cs', $name = 'Category Selector', array(
            'customize_selective_refresh' => true,
        ) );
    }

    /**
     * This is the Widget
     **/
    function widget( $args, $instance ) {

        global $post;

        extract($args);

        // Widget options
        $title = array_key_exists( 'title', $instance ) ? apply_filters('widget_title', $instance['title'] ) : '';
        $this_taxonomy = $instance['taxonomy'] ?? '';
        $show_all = $instance['show_all'] ?? '';
        $hierarchical = !empty( $instance['hierarchical'] ) ? '1' : '0';
        $inv_empty = !empty( $instance['empty'] ) ? '0' : '1'; // invert to go from UI's "show empty" to WP's "hide empty"
        $showcount = !empty( $instance['count'] ) ? '1' : '0';
        $orderby = $instance['orderby'] ?? 'count';
        $ascdsc = $instance['ascdsc'] ?? 'desc';
        $exclude = $instance['exclude'] ?? '';
        $childof = $instance['childof'] ?? '';
        $dropdown = $instance['dropdown'] ?? false;

        // Dropdown doesn't work for built-in taxonomies.
        $builtin = array( 'post_tag', 'post_format', 'category' );

        if ( $dropdown && in_array( $this_taxonomy, $builtin ) ) {
            $dropdown = false;
        }

        // Output
        $tax = $this_taxonomy;

        echo $before_widget;
        echo '<div id="dcs-widget-'.$tax.'-container" class="dojo-category-selector-widget">';

        if ( $title ) echo $before_title . $title . $after_title;

        if($dropdown) {
            $this->_dropdown_view($tax, $show_all, $ascdsc, $showcount, $inv_empty, $childof, $exclude, $hierarchical);
        }
        else {
            $this->_list_view($tax, $show_all, $orderby, $ascdsc, $showcount, $inv_empty, $childof, $exclude, $hierarchical);
        }
        echo '</div>';
        echo $after_widget;
    }


    private function _dropdown_view($tax, $show_all, $ascdsc, $showcount, $inv_empty, $childof, $exclude, $hierarchical): void
    {
        $taxonomy_object = get_taxonomy($tax);

        $reg_walker_taxonomies = ['category', 'post_tag', 'post_format'];
        $walker = in_array($tax, $reg_walker_taxonomies, true) ? '' : new dcswidget_Taxonomy_Dropdown_Walker();


        $args = array(
            'show_option_all'    => $show_all,
            'show_option_none'   => '',
            'orderby'            => 'RANDOM()',
            'order'              => $ascdsc,
            'show_count'         => $showcount,
            'hide_empty'         => $inv_empty,
            'child_of'           => $childof,
            'exclude'            => $exclude,
            'echo'               => 1,
            'hierarchical'       => $hierarchical,
            'name'               => $taxonomy_object->query_var,
            'id'                 => 'dcs-widget-'.$tax,
            'depth'              => 0,
            'taxonomy'           => $tax,
            'hide_if_empty'      => true,
            'walker'			=> $walker,
        );

        echo '<form action="'. get_bloginfo('url'). '" method="get">';
        wp_dropdown_categories($args);
        echo '<input type="submit" value="go &raquo;" /></form>';
    }



    private function _list_view($tax, $show_all, $orderby, $ascdsc, $showcount, $inv_empty, $childof, $exclude, $hierarchical) {

        $args = array(
            'show_option_all'    => $show_all,
            'orderby'            => $orderby,
            'order'              => $ascdsc,
            'style'              => 'list',
            'show_count'         => $showcount,
            'hide_empty'         => $inv_empty,
            'use_desc_for_title' => 0,
            'child_of'           => $childof,
            'exclude'            => $exclude,
            'hierarchical'       => $hierarchical,
            'title_li'           => '',
            'show_option_none'   => 'No Categories',
            'number'             => null,
            'echo'               => 1,
            'depth'              => 0,
            'taxonomy'           => $tax
        );

        if ($tax === 'event_type') {

            $event_type_terms = get_terms( array(
                'taxonomy' => 'event_type',
                'hide_empty' => true,
            ));

            $today = date("Y-m-d",mktime(0,0,0,date("m"),date("d"),date("Y")));
            $args['exclude'] = [];

            foreach ($event_type_terms as $term) {
                $query = new WP_Query(array(
                    'meta_key' => 'event_date',
                    'meta_value' => $today,
                    'meta_compare' => '>=',
                    'tax_query' => [
                        [
                            'taxonomy' => 'event_type',
                            'field' => 'term_id',
                            'terms' => $term->term_id
                        ],
                    ],
                    'posts_per_page' => 1,
                    'fields' => 'ids'
                ));

                if ((!$query->found_posts) > 0) {
                    $args['exclude'][] = $term->term_id;
                }
            }
        }

        echo '<ul id="dcs-widget-'.$tax.'">';
        echo wp_list_categories($args);
        echo '</ul>';
    }



    /** Widget control update */
    function update( $new_instance, $old_instance ) {

        $instance = $old_instance;

        $instance['title']  = strip_tags( $new_instance['title'] );
        $instance['taxonomy'] = strip_tags( $new_instance['taxonomy'] );
        $instance['show_all'] = $new_instance['show_all'];
        $instance['orderby'] = $new_instance['orderby'];
        $instance['ascdsc'] = $new_instance['ascdsc'];
        $instance['exclude'] = $new_instance['exclude'];
        $instance['expandoptions'] = $new_instance['expandoptions'];
        $instance['childof'] = $new_instance['childof'];
        $instance['hierarchical'] = !empty($new_instance['hierarchical']) ? 1 : 0;
        $instance['empty'] = !empty($new_instance['empty']) ? 1 : 0;
        $instance['count'] = !empty($new_instance['count']) ? 1 : 0;
        $instance['dropdown'] = !empty($new_instance['dropdown']) ? 1 : 0;

        return $instance;
    }

    /**
     * Widget settings
     **/
    function form( $instance ) {
        //for showing/hiding advanced options; wordpress moves this script to where it needs to go
        wp_enqueue_script('jquery');
        ?><script>
            jQuery(document).ready(function(){
                var status = jQuery('#<?php echo $this->get_field_id('expandoptions'); ?>').val();
                if ( status === 'expand' ) {
                    //jQuery('.dcsw-expand-options').hide();
                    jQuery('.dcsw-all-options').show();
                } else {
                    jQuery('.dcsw-all-options').hide();
                }
            });
            function dcswExpand(id){
                jQuery('#' + id).val('expand');
                jQuery('.dcsw-all-options').show(500);
                jQuery('.dcsw-expand-options').hide(500);
            }
            function dcswContract(id){
                jQuery('#' + id).val('contract');
                jQuery('.dcsw-all-options').hide(500);
                jQuery('.dcsw-expand-options').show(500);
            }
        </script><?php
        // instance exist? if not set defaults
        if ( $instance ) {
            $title  = $instance['title'];
            $this_taxonomy = $instance['taxonomy'];
            $show_all = $instance['show_all'];
            $orderby = $instance['orderby'];
            $ascdsc = $instance['ascdsc'];
            $exclude = $instance['exclude'];
            $expandoptions = $instance['expandoptions'];
            $childof = $instance['childof'];
            $showcount = isset($instance['count']) ? (bool) $instance['count'] :false;
            $hierarchical = isset( $instance['hierarchical'] ) ? (bool) $instance['hierarchical'] : false;
            $empty = isset( $instance['empty'] ) ? (bool) $instance['empty'] : false;
            $dropdown = isset( $instance['dropdown'] ) ? (bool) $instance['dropdown'] : false;
        } else {
            //These are our defaults
            $title  = '';
            $show_all = '';
            $orderby  = 'name';
            $ascdsc  = 'asc';
            $exclude  = '';
            $expandoptions  = 'contract';
            $childof  = '';
            $this_taxonomy = 'category';//this will display the category taxonomy, which is used for normal, built-in posts
            $hierarchical = false;
            $showcount = false;
            $empty = false;
            $dropdown = false;
        }

        // The widget form ?>
        <p>
            <label for="<?php echo $this->get_field_id('title'); ?>"><?php echo __( 'Title:' ); ?></label>
            <input id="<?php echo $this->get_field_id('title'); ?>" name="<?php echo $this->get_field_name('title'); ?>" type="text" value="<?php echo $title; ?>" class="widefat" />
        </p>
        <p>
            <label for="<?php echo $this->get_field_id('taxonomy'); ?>"><?php echo __( 'Select Taxonomy:' ); ?></label>
            <select name="<?php echo $this->get_field_name('taxonomy'); ?>" id="<?php echo $this->get_field_id('taxonomy'); ?>" class="widefat" style="height: auto;" size="4">
                <?php
                $args=array(
                    'public'   => true,
                    '_builtin' => false //these are manually added to the array later
                );
                $output = 'names'; // or objects
                $operator = 'and'; // 'and' or 'or'
                $taxonomies=get_taxonomies($args,$output,$operator);
                $taxonomies[] = 'category';
                $taxonomies[] = 'post_tag';
                $taxonomies[] = 'post_format';
                foreach ($taxonomies as $taxonomy ) { ?>
                    <option value="<?php echo $taxonomy; ?>" <?php if( $taxonomy == $this_taxonomy ) { echo 'selected="selected"'; } ?>><?php echo $taxonomy;?></option>
                <?php }	?>
            </select>
        </p>
        <h4 class="dcsw-expand-options" style="display: block"><a href="javascript:void(0)" onclick="dcswExpand('<?php echo $this->get_field_id('expandoptions'); ?>')" >More Options...</a></h4>
        <div class="dcsw-all-options">
            <h4 class="dcsw-contract-options"><a href="javascript:void(0)" onclick="dcswContract('<?php echo $this->get_field_id('expandoptions'); ?>')" >Hide Extended Options</a></h4>
            <input type="hidden" value="<?php echo $expandoptions; ?>" id="<?php echo $this->get_field_id('expandoptions'); ?>" name="<?php echo $this->get_field_name('expandoptions'); ?>" />

            <p>
                <label for="<?php echo $this->get_field_id('show_all'); ?>">Show All Text</label><br/>
                <input type="text" class="widefat" name="<?php echo $this->get_field_name('show_all'); ?>" value="<?php echo $show_all; ?>" />
            </p>

            <input type="checkbox" class="checkbox" id="<?php echo $this->get_field_id('count'); ?>" name="<?php echo $this->get_field_name('count'); ?>"<?php checked( $showcount ); ?> />
            <label for="<?php echo $this->get_field_id('count'); ?>"><?php _e( 'Show Post Counts' ); ?></label><br />
            <input type="checkbox" class="checkbox" id="<?php echo $this->get_field_id('hierarchical'); ?>" name="<?php echo $this->get_field_name('hierarchical'); ?>"<?php checked( $hierarchical ); ?> />
            <label for="<?php echo $this->get_field_id('hierarchical'); ?>"><?php _e( 'Show Hierarchy' ); ?></label><br/>
            <input type="checkbox" class="checkbox" id="<?php echo $this->get_field_id('empty'); ?>" name="<?php echo $this->get_field_name('empty'); ?>"<?php checked( $empty ); ?> />
            <label for="<?php echo $this->get_field_id('empty'); ?>"><?php _e( 'Show Empty Terms' ); ?></label></p>

            <p>
                <label for="<?php echo $this->get_field_id('orderby'); ?>"><?php echo __( 'Order By:' ); ?></label>
                <select name="<?php echo $this->get_field_name('orderby'); ?>" id="<?php echo $this->get_field_id('orderby'); ?>" class="widefat" >
                    <option value="ID" <?php if( $orderby == 'ID' ) { echo 'selected="selected"'; } ?>>ID</option>
                    <option value="name" <?php if( $orderby == 'name' ) { echo 'selected="selected"'; } ?>>Name</option>
                    <option value="slug" <?php if( $orderby == 'slug' ) { echo 'selected="selected"'; } ?>>Slug</option>
                    <option value="count" <?php if( $orderby == 'count' ) { echo 'selected="selected"'; } ?>>Count</option>
                    <option value="term_group" <?php if( $orderby == 'term_group' ) { echo 'selected="selected"'; } ?>>Term Group</option>
                </select>
            </p>
            <p>
                <label><input type="radio" name="<?php echo $this->get_field_name('ascdsc'); ?>" value="asc" <?php if( $ascdsc == 'asc' ) { echo 'checked'; } ?>/> Ascending</label><br/>
                <label><input type="radio" name="<?php echo $this->get_field_name('ascdsc'); ?>" value="desc" <?php if( $ascdsc == 'desc' ) { echo 'checked'; } ?>/> Descending</label>
            </p>
            <p>
                <label for="<?php echo $this->get_field_id('exclude'); ?>">Exclude (comma-separated list of ids to exclude)</label><br/>
                <input type="text" class="widefat" name="<?php echo $this->get_field_name('exclude'); ?>" value="<?php echo $exclude; ?>" />
            </p>
            <p>
                <label for="<?php echo $this->get_field_id('exclude'); ?>">Only Show Children of (category id)</label><br/>
                <input type="text" class="widefat" name="<?php echo $this->get_field_name('childof'); ?>" value="<?php echo $childof; ?>" />
            </p>
            <input type="checkbox" class="checkbox" id="<?php echo $this->get_field_id('dropdown'); ?>" name="<?php echo $this->get_field_name('dropdown'); ?>"<?php checked( $dropdown ); ?> />
            <label for="<?php echo $this->get_field_id('dropdown'); ?>"><?php _e( 'Display as Dropdown' ); ?></label></p>
        </div>
        <?php
    }

} // class dojo_cs

/* Custom version of Walker_CategoryDropdown */
class dcswidget_Taxonomy_Dropdown_Walker extends Walker {
    var $tree_type = 'category';
    var $db_fields = array ( 'id' => 'term_id', 'parent' => 'parent' );

    function start_el( &$output, $term, $depth = 0, $args = array(), $current_object_id = 0 ) {
        $term = get_term( $term, $term->taxonomy );
        $term_slug = $term->slug;

        $text = str_repeat( '&nbsp;', $depth * 3 ) . $term->name;
        if ( $args['show_count'] ) {
            $text .= '&nbsp;('. $term->count .')';
        }

        $class_name = 'level-' . $depth;

        $output.= "\t" . '<option' . ' class="' . esc_attr( $class_name ) . '" value="' . esc_attr( $term_slug ) . '">' . esc_html( $text ) . '</option>' . "\n";
    }
}

//Add current-cat class to the 'All' link if not in any other
function wp_list_categories_highlight_all( $output, $args ) {
    if ( array_key_exists( 'show_option_all', $args ) && $args['show_option_all'] ) {
        if ( ! array_key_exists( 'current_category', $args ) || $args['current_category'] ) {
            if ( is_category() || is_tax() || is_tag() ) {
                if ( ! array_key_exists( 'taxonomy', $args ) ) {
                    $args['taxonomy'] = 'category';
                }
                $current_term_object = get_queried_object();
                if ( $args['taxonomy'] !== $current_term_object->taxonomy ) {
                    $output = str_replace( "class='cat-item-all'", "class='cat-item-all current-cat'", $output );
                }
            } else {
                $output = str_replace( "class='cat-item-all'", "class='cat-item-all current-cat'", $output );
            }
        }
    }
    return $output;
}
add_filter( 'wp_list_categories', 'wp_list_categories_highlight_all', 10, 2 );