<?php 
namespace NinjaForms\Stripe\Checkout\Admin;



if (!defined('ABSPATH')) exit;

class VersionCheck
{

  const MINIMUM_REQUIRED_NINJA_FORMS_VERSION = '3.6.30';

  /**
   * Activate checks and notices for plugin's envirnoment compatibility 
   *
   * @return void
   */
  public function activate(): void
  {
    add_action('ninja_forms_loaded', array($this, 'ensureVersionCompatibility'), 0);
  }

  /**
   * If Stripe and NF core versions are incompatible, display user notice
   *
   * NOTE: this does not stop functioning of plugin, but warns user that
   * functionality may be affected
   *
   * @return void
   */
  public function ensureVersionCompatibility(): void
  {
    $versionsAreCompatible = $this->checkVersionCompatibility();

    if ($versionsAreCompatible) {
      return;
    }

    add_filter('nf_admin_notices', [$this, 'addIncompatibleVersionsNotice']);
  }

  /**
   * Check that required versions are installed for proper functionality
   *
   * Default is to pass checks; only fail if known version incompatibility
   * 
   * @return boolean
   */
  protected function checkVersionCompatibility(): bool
  {
    $return = true;

    if (
      \class_exists('Ninja_Forms')
      && '3.0-versionNumberPlaceholder' !== \Ninja_Forms::VERSION
      && version_compare(\Ninja_Forms::VERSION, self::MINIMUM_REQUIRED_NINJA_FORMS_VERSION, '<')
    ) {
      $return = false;
    }
    
    return $return;
  }

  /**
   * Add NF admin notice for incompatible versions
   *
   * @param array $notices
   * @return array
   */
  public function addIncompatibleVersionsNotice($notices): array
  {

    $msg = sprintf(esc_html__('Please update Ninja Forms. The version you are using is not compatible with this version of Stripe.. %sIf you have questions or need help, please %scontact our support team%s.', 'ninja-forms'), '<br />', '<a style="background-color:transparent; padding:0; text-decoration:underline;"href="'. \admin_url('admin.php?page=nf-system-status').'">', '</a>');

    $notices['nfstripe_incompatible_versions'] = array(
      'title' => 'Important',
      'msg' => $msg,
      'int' => 0,
      'link'=>'<a href="'. \admin_url('update-core.php').'">'.'Update now</a>'
    );

    return $notices;
  }
}
