<?php

namespace NinjaForms\StripeCheckout\Common\Traits;

use NinjaForms\StripeCheckout\Common\VendorDist\Psr\Log\LoggerInterface;
use NinjaForms\StripeCheckout\Common\VendorDist\Psr\Log\NullLogger;

trait Logger
{

    /** @var LoggerInterface */
    protected $logger;

    /** 
     * Array construct of LogEntry entity
     * 
     * @var array 
     */
    protected $logEntryArray;

    /**
     * Array of supporting data
     *  
     * @var array 
     */
    protected $supportingData;

    /**
     * Set the logger
     *
     * @return  self
     */
    public function setLogger($logger): self
    {
        $this->logger = $logger;

        return $this;
    }

    /**
     * Provide the logger, fallback to empty logger
     */
    protected function getLogger()
    {
        if (!isset($this->logger)) {
            $this->logger = new NullLogger();
        }

        return $this->logger;
    }

    /**
     * Construct populated logEntryArray and supportingData properties
     *
     * Optional to add summary and initialize array of supporting data
     *
     * @param int|null $backtraceSteps
     * @param string|null $summary
     * @param array|null $supportingData
     * @return void
     */
    public function initializeLogEntry(?int $backtraceSteps = 5, ?string $summary = '', ?array $supportingData = []): void
    {
        if ($backtraceSteps < 3) {
            $backtraceSteps = 3;
        }

        $debugBacktrace = array_reverse($this->getBacktrace($backtraceSteps));

        $logPointIndex = $backtraceSteps - 3;
        $logPoint = $this->constructLogPoint($debugBacktrace[$logPointIndex]);

        $i = 0;
        $callingFunctions = [];

        while ($i < $logPointIndex) {
            if (!isset($debugBacktrace[$i])) {
                continue;
            }

            $callingFunctions[] = $this->constructLogPoint($debugBacktrace[$i]);
            $i++;
        }

        $this->supportingData = $supportingData;

        $this->supportingData['callingFunctions'] = $callingFunctions;

        $this->logEntryArray = [
            'logPoint' => $logPoint,
            'timestamp' => time(),
            'summary' => $summary,
            'supportingData' => json_encode($this->supportingData)
        ];

    }

    /**
     * Provide a backtrace three calls deep
     *
     * @param int $backtraceSteps
     * @return array
     */
    protected function getBacktrace(?int $backtraceSteps = 5): array
    {
        $debugBacktrace = \debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS, $backtraceSteps);

        return $debugBacktrace;
    }

    /**
     * Construct the logpoint from the backtrace data
     *
     * @param array $debugBacktrace
     * @return string
     */
    protected function constructLogPoint(array $debugBacktrace): string
    {
        $logpoint = '';

        if (isset($debugBacktrace['class'])) {
            $logpoint .= $debugBacktrace['class'];
        }

        if (isset($debugBacktrace['type'])) {
            $logpoint .= $debugBacktrace['type'];
        } else {
            $logpoint .= ' ';
        }
        if (isset($debugBacktrace['function'])) {
            $logpoint .= $debugBacktrace['function'];
        }

        return $logpoint;
    }

    /** @return void */
    public function finalizeLogEntry(): void
    {
        // First, add supporting data to log entry     
        $this->logEntryArray['supportingData'] = json_encode($this->supportingData);

        // Then log it as debug entry
        $this->getLogger()->debug($this->logEntryArray['summary'], $this->logEntryArray);
    }
}
