<?php

namespace NinjaForms\StripeCheckout\Common\Routes;

if (!defined('ABSPATH')) exit;

use NinjaForms\StripeCheckout\Common\Interfaces\NfLogHandler;
use NinjaForms\StripeCheckout\Common\Handlers\DownloadDebugLog;
use \WP_REST_Request;

/**
 * 
 */
class DebugLog
{
    const GETLOGSENDPOINT = 'debug-log/get-all';
    const DELETELOGSENDPOINT = 'debug-log/delete-all';
    
    /**
     * Route upon which endpoints are built
     *
     * @var string
     */
    protected $route;

    /** @var NfLogHandler */
    protected $logHandler;

    /** @var DownloadDebugLog */
    protected $downloadDebugLog;

    public function __construct(NfLogHandler $logHandler, string $route, DownloadDebugLog $downloadDebugLog)
    {
        $this->logHandler = $logHandler;
        $this->route = $route;
        $this->downloadDebugLog = $downloadDebugLog;

        \add_action('rest_api_init', [$this, 'registerRoutes']);
    }
    /**
     * Register REST API routes for the debug log
     */
    function registerRoutes()
    {
        \register_rest_route($this->route, self::GETLOGSENDPOINT, array(
            'methods' => 'POST',
            'args' => [
                '_wpnonce'=>[
                    'type'=>'string',
                    'required'=>true
                ]
            ],
            'callback' => [$this, 'getDebugLogEntries'],
            'permission_callback' => [$this, 'getDebugLogEntriesPermissionCallback']
        ));

        \register_rest_route($this->route, self::DELETELOGSENDPOINT, array(
            'methods' => 'POST',
            'args' => [
                '_wpnonce'=>[
                    'type'=>'string',
                    'required'=>true
                ]
            ],
            'callback' => [$this, 'deleteAllDiagnostics'],
            'permission_callback' => [$this, 'getDebugLogEntriesPermissionCallback']
        ));
    }

    public function getDebugLogEntries(WP_REST_Request $request)
    {
        $returnedNonce = $request->get_param('_wpnonce');

        $authorized = \wp_verify_nonce($returnedNonce, 'wp_rest');

        if ($authorized) {

            $data = $this->downloadDebugLog->getDebugJson();
        } else {
            $data = __('Nonce failed for authorization', 'ninja-forms-stripe');
        }

        $response = [
            'data' => $data
        ];

        return \rest_ensure_response($response);
    }

    public function deleteAllDiagnostics(WP_REST_Request $request)
    {
        $returnedNonce = $request->get_param('_wpnonce');

        $authorized = \wp_verify_nonce($returnedNonce, 'wp_rest');

        if ($authorized) {
            $this->logHandler->deletePluginLogEntries();

            $result = __('Request made to delete all plugin log entries', 'ninja-forms-stripe');
        } else {
            $result = __('Nonce failed for authorization', 'ninja-forms-stripe');
        }

        $response = [
            'result' => $result
        ];

        return \rest_ensure_response($response);
    }

    /**
     * Verify user is permitted to download diagnostics
     *
     * Authorization handled using WP nonce
     * @param WP_REST_Request $request
     * @return void
     */
    public function getDebugLogEntriesPermissionCallback(WP_REST_Request $request)
    {
        $default = \current_user_can("delete_others_posts");

        $return = \apply_filters($this->route.'_debug_rest_endpoint_permissions',$default);

        return $return;
    }
}
