<?php

declare(strict_types=1);

namespace Square\Models;

/**
 * Contains details on how to fulfill this order.
 */
class OrderFulfillment implements \JsonSerializable
{
    /**
     * @var string|null
     */
    private $uid;

    /**
     * @var string|null
     */
    private $type;

    /**
     * @var string|null
     */
    private $state;

    /**
     * @var array|null
     */
    private $metadata;

    /**
     * @var OrderFulfillmentPickupDetails|null
     */
    private $pickupDetails;

    /**
     * @var OrderFulfillmentShipmentDetails|null
     */
    private $shipmentDetails;

    /**
     * Returns Uid.
     *
     * Unique ID that identifies the fulfillment only within this order.
     */
    public function getUid(): ?string
    {
        return $this->uid;
    }

    /**
     * Sets Uid.
     *
     * Unique ID that identifies the fulfillment only within this order.
     *
     * @maps uid
     */
    public function setUid(?string $uid): void
    {
        $this->uid = $uid;
    }

    /**
     * Returns Type.
     *
     * The type of fulfillment.
     */
    public function getType(): ?string
    {
        return $this->type;
    }

    /**
     * Sets Type.
     *
     * The type of fulfillment.
     *
     * @maps type
     */
    public function setType(?string $type): void
    {
        $this->type = $type;
    }

    /**
     * Returns State.
     *
     * The current state of this fulfillment.
     */
    public function getState(): ?string
    {
        return $this->state;
    }

    /**
     * Sets State.
     *
     * The current state of this fulfillment.
     *
     * @maps state
     */
    public function setState(?string $state): void
    {
        $this->state = $state;
    }

    /**
     * Returns Metadata.
     *
     * Application-defined data attached to this fulfillment. Metadata fields are intended
     * to store descriptive references or associations with an entity in another system or store brief
     * information about the object. Square does not process this field; it only stores and returns it
     * in relevant API calls. Do not use metadata to store any sensitive information (personally
     * identifiable information, card details, etc.).
     *
     * Keys written by applications must be 60 characters or less and must be in the character set
     * `[a-zA-Z0-9_-]`. Entries may also include metadata generated by Square. These keys are prefixed
     * with a namespace, separated from the key with a ':' character.
     *
     * Values have a max length of 255 characters.
     *
     * An application may have up to 10 entries per metadata field.
     *
     * Entries written by applications are private and can only be read or modified by the same
     * application.
     *
     * See [Metadata](https://developer.squareup.com/docs/build-basics/metadata) for more information.
     */
    public function getMetadata(): ?array
    {
        return $this->metadata;
    }

    /**
     * Sets Metadata.
     *
     * Application-defined data attached to this fulfillment. Metadata fields are intended
     * to store descriptive references or associations with an entity in another system or store brief
     * information about the object. Square does not process this field; it only stores and returns it
     * in relevant API calls. Do not use metadata to store any sensitive information (personally
     * identifiable information, card details, etc.).
     *
     * Keys written by applications must be 60 characters or less and must be in the character set
     * `[a-zA-Z0-9_-]`. Entries may also include metadata generated by Square. These keys are prefixed
     * with a namespace, separated from the key with a ':' character.
     *
     * Values have a max length of 255 characters.
     *
     * An application may have up to 10 entries per metadata field.
     *
     * Entries written by applications are private and can only be read or modified by the same
     * application.
     *
     * See [Metadata](https://developer.squareup.com/docs/build-basics/metadata) for more information.
     *
     * @maps metadata
     */
    public function setMetadata(?array $metadata): void
    {
        $this->metadata = $metadata;
    }

    /**
     * Returns Pickup Details.
     *
     * Contains details necessary to fulfill a pickup order.
     */
    public function getPickupDetails(): ?OrderFulfillmentPickupDetails
    {
        return $this->pickupDetails;
    }

    /**
     * Sets Pickup Details.
     *
     * Contains details necessary to fulfill a pickup order.
     *
     * @maps pickup_details
     */
    public function setPickupDetails(?OrderFulfillmentPickupDetails $pickupDetails): void
    {
        $this->pickupDetails = $pickupDetails;
    }

    /**
     * Returns Shipment Details.
     *
     * Contains details necessary to fulfill a shipment order.
     */
    public function getShipmentDetails(): ?OrderFulfillmentShipmentDetails
    {
        return $this->shipmentDetails;
    }

    /**
     * Sets Shipment Details.
     *
     * Contains details necessary to fulfill a shipment order.
     *
     * @maps shipment_details
     */
    public function setShipmentDetails(?OrderFulfillmentShipmentDetails $shipmentDetails): void
    {
        $this->shipmentDetails = $shipmentDetails;
    }

    /**
     * Encode this object to JSON
     *
     * @return mixed
     */
    public function jsonSerialize()
    {
        $json = [];
        $json['uid']             = $this->uid;
        $json['type']            = $this->type;
        $json['state']           = $this->state;
        $json['metadata']        = $this->metadata;
        $json['pickup_details']  = $this->pickupDetails;
        $json['shipment_details'] = $this->shipmentDetails;

        return array_filter($json, function ($val) {
            return $val !== null;
        });
    }
}
