<?php

declare(strict_types=1);

namespace Square\Models;

/**
 * Creates a payment from the source (nonce, card on file, etc.)
 *
 * The `PAYMENTS_WRITE_ADDITIONAL_RECIPIENTS` OAuth permission is required
 * to enable application fees.
 *
 * For more information, see [Payments and Refunds Overview](https://developer.squareup.
 * com/docs/payments-api/overview).
 *
 * For information about application fees in a payment, see
 * [Collect Fees](https://developer.squareup.com/docs/payments-api/take-payments-and-collect-fees).
 */
class CreatePaymentRequest implements \JsonSerializable
{
    /**
     * @var string
     */
    private $sourceId;

    /**
     * @var string
     */
    private $idempotencyKey;

    /**
     * @var Money
     */
    private $amountMoney;

    /**
     * @var Money|null
     */
    private $tipMoney;

    /**
     * @var Money|null
     */
    private $appFeeMoney;

    /**
     * @var string|null
     */
    private $delayDuration;

    /**
     * @var bool|null
     */
    private $autocomplete;

    /**
     * @var string|null
     */
    private $orderId;

    /**
     * @var string|null
     */
    private $customerId;

    /**
     * @var string|null
     */
    private $locationId;

    /**
     * @var string|null
     */
    private $referenceId;

    /**
     * @var string|null
     */
    private $verificationToken;

    /**
     * @var bool|null
     */
    private $acceptPartialAuthorization;

    /**
     * @var string|null
     */
    private $buyerEmailAddress;

    /**
     * @var Address|null
     */
    private $billingAddress;

    /**
     * @var Address|null
     */
    private $shippingAddress;

    /**
     * @var string|null
     */
    private $note;

    /**
     * @var string|null
     */
    private $statementDescriptionIdentifier;

    /**
     * @param string $sourceId
     * @param string $idempotencyKey
     * @param Money $amountMoney
     */
    public function __construct(string $sourceId, string $idempotencyKey, Money $amountMoney)
    {
        $this->sourceId = $sourceId;
        $this->idempotencyKey = $idempotencyKey;
        $this->amountMoney = $amountMoney;
    }

    /**
     * Returns Source Id.
     *
     * The ID for the source of funds for this payment.  This can be a nonce
     * generated by the Payment Form or a card on file made with the Customers API.
     */
    public function getSourceId(): string
    {
        return $this->sourceId;
    }

    /**
     * Sets Source Id.
     *
     * The ID for the source of funds for this payment.  This can be a nonce
     * generated by the Payment Form or a card on file made with the Customers API.
     *
     * @required
     * @maps source_id
     */
    public function setSourceId(string $sourceId): void
    {
        $this->sourceId = $sourceId;
    }

    /**
     * Returns Idempotency Key.
     *
     * A unique string that identifies this CreatePayment request. Keys can be any valid string
     * but must be unique for every CreatePayment request.
     *
     * Max: 45 characters
     *
     * See [Idempotency keys](https://developer.squareup.com/docs/basics/api101/idempotency) for more
     * information.
     */
    public function getIdempotencyKey(): string
    {
        return $this->idempotencyKey;
    }

    /**
     * Sets Idempotency Key.
     *
     * A unique string that identifies this CreatePayment request. Keys can be any valid string
     * but must be unique for every CreatePayment request.
     *
     * Max: 45 characters
     *
     * See [Idempotency keys](https://developer.squareup.com/docs/basics/api101/idempotency) for more
     * information.
     *
     * @required
     * @maps idempotency_key
     */
    public function setIdempotencyKey(string $idempotencyKey): void
    {
        $this->idempotencyKey = $idempotencyKey;
    }

    /**
     * Returns Amount Money.
     *
     * Represents an amount of money. `Money` fields can be signed or unsigned.
     * Fields that do not explicitly define whether they are signed or unsigned are
     * considered unsigned and can only hold positive amounts. For signed fields, the
     * sign of the value indicates the purpose of the money transfer. See
     * [Working with Monetary Amounts](https://developer.squareup.com/docs/build-basics/working-with-
     * monetary-amounts)
     * for more information.
     */
    public function getAmountMoney(): Money
    {
        return $this->amountMoney;
    }

    /**
     * Sets Amount Money.
     *
     * Represents an amount of money. `Money` fields can be signed or unsigned.
     * Fields that do not explicitly define whether they are signed or unsigned are
     * considered unsigned and can only hold positive amounts. For signed fields, the
     * sign of the value indicates the purpose of the money transfer. See
     * [Working with Monetary Amounts](https://developer.squareup.com/docs/build-basics/working-with-
     * monetary-amounts)
     * for more information.
     *
     * @required
     * @maps amount_money
     */
    public function setAmountMoney(Money $amountMoney): void
    {
        $this->amountMoney = $amountMoney;
    }

    /**
     * Returns Tip Money.
     *
     * Represents an amount of money. `Money` fields can be signed or unsigned.
     * Fields that do not explicitly define whether they are signed or unsigned are
     * considered unsigned and can only hold positive amounts. For signed fields, the
     * sign of the value indicates the purpose of the money transfer. See
     * [Working with Monetary Amounts](https://developer.squareup.com/docs/build-basics/working-with-
     * monetary-amounts)
     * for more information.
     */
    public function getTipMoney(): ?Money
    {
        return $this->tipMoney;
    }

    /**
     * Sets Tip Money.
     *
     * Represents an amount of money. `Money` fields can be signed or unsigned.
     * Fields that do not explicitly define whether they are signed or unsigned are
     * considered unsigned and can only hold positive amounts. For signed fields, the
     * sign of the value indicates the purpose of the money transfer. See
     * [Working with Monetary Amounts](https://developer.squareup.com/docs/build-basics/working-with-
     * monetary-amounts)
     * for more information.
     *
     * @maps tip_money
     */
    public function setTipMoney(?Money $tipMoney): void
    {
        $this->tipMoney = $tipMoney;
    }

    /**
     * Returns App Fee Money.
     *
     * Represents an amount of money. `Money` fields can be signed or unsigned.
     * Fields that do not explicitly define whether they are signed or unsigned are
     * considered unsigned and can only hold positive amounts. For signed fields, the
     * sign of the value indicates the purpose of the money transfer. See
     * [Working with Monetary Amounts](https://developer.squareup.com/docs/build-basics/working-with-
     * monetary-amounts)
     * for more information.
     */
    public function getAppFeeMoney(): ?Money
    {
        return $this->appFeeMoney;
    }

    /**
     * Sets App Fee Money.
     *
     * Represents an amount of money. `Money` fields can be signed or unsigned.
     * Fields that do not explicitly define whether they are signed or unsigned are
     * considered unsigned and can only hold positive amounts. For signed fields, the
     * sign of the value indicates the purpose of the money transfer. See
     * [Working with Monetary Amounts](https://developer.squareup.com/docs/build-basics/working-with-
     * monetary-amounts)
     * for more information.
     *
     * @maps app_fee_money
     */
    public function setAppFeeMoney(?Money $appFeeMoney): void
    {
        $this->appFeeMoney = $appFeeMoney;
    }

    /**
     * Returns Delay Duration.
     *
     * The duration of time after the payment's creation when Square automatically cancels the
     * payment. This automatic cancellation applies only to payments that don't reach a terminal state
     * (COMPLETED, CANCELED, or FAILED) before the `delay_duration` time period.
     *
     * This parameter should be specified as a time duration, in RFC 3339 format, with a minimum value
     * of 1 minute.
     *
     * Notes:
     * This feature is only supported for card payments. This parameter can only be set for a delayed
     * capture payment (`autocomplete=false`).
     *
     * Default:
     *
     * - Card Present payments: "PT36H" (36 hours) from the creation time.
     * - Card Not Present payments: "P7D" (7 days) from the creation time.
     */
    public function getDelayDuration(): ?string
    {
        return $this->delayDuration;
    }

    /**
     * Sets Delay Duration.
     *
     * The duration of time after the payment's creation when Square automatically cancels the
     * payment. This automatic cancellation applies only to payments that don't reach a terminal state
     * (COMPLETED, CANCELED, or FAILED) before the `delay_duration` time period.
     *
     * This parameter should be specified as a time duration, in RFC 3339 format, with a minimum value
     * of 1 minute.
     *
     * Notes:
     * This feature is only supported for card payments. This parameter can only be set for a delayed
     * capture payment (`autocomplete=false`).
     *
     * Default:
     *
     * - Card Present payments: "PT36H" (36 hours) from the creation time.
     * - Card Not Present payments: "P7D" (7 days) from the creation time.
     *
     * @maps delay_duration
     */
    public function setDelayDuration(?string $delayDuration): void
    {
        $this->delayDuration = $delayDuration;
    }

    /**
     * Returns Autocomplete.
     *
     * If set to `true`, this payment will be completed when possible. If
     * set to `false`, this payment will be held in an approved state until either
     * explicitly completed (captured) or canceled (voided). For more information, see
     * [Delayed Payments](https://developer.squareup.com/docs/payments-api/take-payments#delayed-payments).
     *
     * Default: true
     */
    public function getAutocomplete(): ?bool
    {
        return $this->autocomplete;
    }

    /**
     * Sets Autocomplete.
     *
     * If set to `true`, this payment will be completed when possible. If
     * set to `false`, this payment will be held in an approved state until either
     * explicitly completed (captured) or canceled (voided). For more information, see
     * [Delayed Payments](https://developer.squareup.com/docs/payments-api/take-payments#delayed-payments).
     *
     * Default: true
     *
     * @maps autocomplete
     */
    public function setAutocomplete(?bool $autocomplete): void
    {
        $this->autocomplete = $autocomplete;
    }

    /**
     * Returns Order Id.
     *
     * Associate a previously created order with this payment
     */
    public function getOrderId(): ?string
    {
        return $this->orderId;
    }

    /**
     * Sets Order Id.
     *
     * Associate a previously created order with this payment
     *
     * @maps order_id
     */
    public function setOrderId(?string $orderId): void
    {
        $this->orderId = $orderId;
    }

    /**
     * Returns Customer Id.
     *
     * The [Customer](#type-customer) ID of the customer associated with the payment.
     * Required if the `source_id` refers to a card on file created using the Customers API.
     */
    public function getCustomerId(): ?string
    {
        return $this->customerId;
    }

    /**
     * Sets Customer Id.
     *
     * The [Customer](#type-customer) ID of the customer associated with the payment.
     * Required if the `source_id` refers to a card on file created using the Customers API.
     *
     * @maps customer_id
     */
    public function setCustomerId(?string $customerId): void
    {
        $this->customerId = $customerId;
    }

    /**
     * Returns Location Id.
     *
     * The location ID to associate with the payment. If not specified, the default location is
     * used.
     */
    public function getLocationId(): ?string
    {
        return $this->locationId;
    }

    /**
     * Sets Location Id.
     *
     * The location ID to associate with the payment. If not specified, the default location is
     * used.
     *
     * @maps location_id
     */
    public function setLocationId(?string $locationId): void
    {
        $this->locationId = $locationId;
    }

    /**
     * Returns Reference Id.
     *
     * A user-defined ID to associate with the payment.
     * You can use this field to associate the payment to an entity in an external system.
     * For example, you might specify an order ID that is generated by a third-party shopping cart.
     *
     * Limit 40 characters.
     */
    public function getReferenceId(): ?string
    {
        return $this->referenceId;
    }

    /**
     * Sets Reference Id.
     *
     * A user-defined ID to associate with the payment.
     * You can use this field to associate the payment to an entity in an external system.
     * For example, you might specify an order ID that is generated by a third-party shopping cart.
     *
     * Limit 40 characters.
     *
     * @maps reference_id
     */
    public function setReferenceId(?string $referenceId): void
    {
        $this->referenceId = $referenceId;
    }

    /**
     * Returns Verification Token.
     *
     * An identifying token generated by `SqPaymentForm.verifyBuyer()`.
     * Verification tokens encapsulate customer device information and 3-D Secure
     * challenge results to indicate that Square has verified the buyer identity.
     *
     * See the [SCA Overview](https://developer.squareup.com/docs/sca-overview).
     */
    public function getVerificationToken(): ?string
    {
        return $this->verificationToken;
    }

    /**
     * Sets Verification Token.
     *
     * An identifying token generated by `SqPaymentForm.verifyBuyer()`.
     * Verification tokens encapsulate customer device information and 3-D Secure
     * challenge results to indicate that Square has verified the buyer identity.
     *
     * See the [SCA Overview](https://developer.squareup.com/docs/sca-overview).
     *
     * @maps verification_token
     */
    public function setVerificationToken(?string $verificationToken): void
    {
        $this->verificationToken = $verificationToken;
    }

    /**
     * Returns Accept Partial Authorization.
     *
     * If set to true and charging a Square Gift Card, a payment may be returned with
     * amount_money equal to less than what was requested.  Example, a request for $20 when charging
     * a Square Gift Card with balance of $5 wil result in an APPROVED payment of $5.  You may choose
     * to prompt the buyer for an additional payment to cover the remainder, or cancel the gift card
     * payment.  Cannot be `true` when `autocomplete = true`.
     *
     * For more information, see
     * [Partial amount with Square gift cards](https://developer.squareup.com/docs/payments-api/take-
     * payments#partial-payment-gift-card).
     *
     * Default: false
     */
    public function getAcceptPartialAuthorization(): ?bool
    {
        return $this->acceptPartialAuthorization;
    }

    /**
     * Sets Accept Partial Authorization.
     *
     * If set to true and charging a Square Gift Card, a payment may be returned with
     * amount_money equal to less than what was requested.  Example, a request for $20 when charging
     * a Square Gift Card with balance of $5 wil result in an APPROVED payment of $5.  You may choose
     * to prompt the buyer for an additional payment to cover the remainder, or cancel the gift card
     * payment.  Cannot be `true` when `autocomplete = true`.
     *
     * For more information, see
     * [Partial amount with Square gift cards](https://developer.squareup.com/docs/payments-api/take-
     * payments#partial-payment-gift-card).
     *
     * Default: false
     *
     * @maps accept_partial_authorization
     */
    public function setAcceptPartialAuthorization(?bool $acceptPartialAuthorization): void
    {
        $this->acceptPartialAuthorization = $acceptPartialAuthorization;
    }

    /**
     * Returns Buyer Email Address.
     *
     * The buyer's e-mail address
     */
    public function getBuyerEmailAddress(): ?string
    {
        return $this->buyerEmailAddress;
    }

    /**
     * Sets Buyer Email Address.
     *
     * The buyer's e-mail address
     *
     * @maps buyer_email_address
     */
    public function setBuyerEmailAddress(?string $buyerEmailAddress): void
    {
        $this->buyerEmailAddress = $buyerEmailAddress;
    }

    /**
     * Returns Billing Address.
     *
     * Represents a physical address.
     */
    public function getBillingAddress(): ?Address
    {
        return $this->billingAddress;
    }

    /**
     * Sets Billing Address.
     *
     * Represents a physical address.
     *
     * @maps billing_address
     */
    public function setBillingAddress(?Address $billingAddress): void
    {
        $this->billingAddress = $billingAddress;
    }

    /**
     * Returns Shipping Address.
     *
     * Represents a physical address.
     */
    public function getShippingAddress(): ?Address
    {
        return $this->shippingAddress;
    }

    /**
     * Sets Shipping Address.
     *
     * Represents a physical address.
     *
     * @maps shipping_address
     */
    public function setShippingAddress(?Address $shippingAddress): void
    {
        $this->shippingAddress = $shippingAddress;
    }

    /**
     * Returns Note.
     *
     * An optional note to be entered by the developer when creating a payment
     *
     * Limit 500 characters.
     */
    public function getNote(): ?string
    {
        return $this->note;
    }

    /**
     * Sets Note.
     *
     * An optional note to be entered by the developer when creating a payment
     *
     * Limit 500 characters.
     *
     * @maps note
     */
    public function setNote(?string $note): void
    {
        $this->note = $note;
    }

    /**
     * Returns Statement Description Identifier.
     *
     * Optional additional payment information to include on the customer's card statement
     * as part of statement description. This can be, for example, an invoice number, ticket number,
     * or short description that uniquely identifies the purchase.
     *
     * Note that the `statement_description_identifier` may get truncated on the statement description
     * to fit the required information including the Square identifier (SQ *) and name of the
     * merchant taking the payment.
     */
    public function getStatementDescriptionIdentifier(): ?string
    {
        return $this->statementDescriptionIdentifier;
    }

    /**
     * Sets Statement Description Identifier.
     *
     * Optional additional payment information to include on the customer's card statement
     * as part of statement description. This can be, for example, an invoice number, ticket number,
     * or short description that uniquely identifies the purchase.
     *
     * Note that the `statement_description_identifier` may get truncated on the statement description
     * to fit the required information including the Square identifier (SQ *) and name of the
     * merchant taking the payment.
     *
     * @maps statement_description_identifier
     */
    public function setStatementDescriptionIdentifier(?string $statementDescriptionIdentifier): void
    {
        $this->statementDescriptionIdentifier = $statementDescriptionIdentifier;
    }

    /**
     * Encode this object to JSON
     *
     * @return mixed
     */
    public function jsonSerialize()
    {
        $json = [];
        $json['source_id']                      = $this->sourceId;
        $json['idempotency_key']                = $this->idempotencyKey;
        $json['amount_money']                   = $this->amountMoney;
        $json['tip_money']                      = $this->tipMoney;
        $json['app_fee_money']                  = $this->appFeeMoney;
        $json['delay_duration']                 = $this->delayDuration;
        $json['autocomplete']                   = $this->autocomplete;
        $json['order_id']                       = $this->orderId;
        $json['customer_id']                    = $this->customerId;
        $json['location_id']                    = $this->locationId;
        $json['reference_id']                   = $this->referenceId;
        $json['verification_token']             = $this->verificationToken;
        $json['accept_partial_authorization']   = $this->acceptPartialAuthorization;
        $json['buyer_email_address']            = $this->buyerEmailAddress;
        $json['billing_address']                = $this->billingAddress;
        $json['shipping_address']               = $this->shippingAddress;
        $json['note']                           = $this->note;
        $json['statement_description_identifier'] = $this->statementDescriptionIdentifier;

        return array_filter($json, function ($val) {
            return $val !== null;
        });
    }
}
