<?php

declare(strict_types=1);

namespace Square\Models;

/**
 * Represents an additional recipient (other than the merchant) receiving a portion of this tender.
 */
class AdditionalRecipient implements \JsonSerializable
{
    /**
     * @var string
     */
    private $locationId;

    /**
     * @var string
     */
    private $description;

    /**
     * @var Money
     */
    private $amountMoney;

    /**
     * @var string|null
     */
    private $receivableId;

    /**
     * @param string $locationId
     * @param string $description
     * @param Money $amountMoney
     */
    public function __construct(string $locationId, string $description, Money $amountMoney)
    {
        $this->locationId = $locationId;
        $this->description = $description;
        $this->amountMoney = $amountMoney;
    }

    /**
     * Returns Location Id.
     *
     * The location ID for a recipient (other than the merchant) receiving a portion of this tender.
     */
    public function getLocationId(): string
    {
        return $this->locationId;
    }

    /**
     * Sets Location Id.
     *
     * The location ID for a recipient (other than the merchant) receiving a portion of this tender.
     *
     * @required
     * @maps location_id
     */
    public function setLocationId(string $locationId): void
    {
        $this->locationId = $locationId;
    }

    /**
     * Returns Description.
     *
     * The description of the additional recipient.
     */
    public function getDescription(): string
    {
        return $this->description;
    }

    /**
     * Sets Description.
     *
     * The description of the additional recipient.
     *
     * @required
     * @maps description
     */
    public function setDescription(string $description): void
    {
        $this->description = $description;
    }

    /**
     * Returns Amount Money.
     *
     * Represents an amount of money. `Money` fields can be signed or unsigned.
     * Fields that do not explicitly define whether they are signed or unsigned are
     * considered unsigned and can only hold positive amounts. For signed fields, the
     * sign of the value indicates the purpose of the money transfer. See
     * [Working with Monetary Amounts](https://developer.squareup.com/docs/build-basics/working-with-
     * monetary-amounts)
     * for more information.
     */
    public function getAmountMoney(): Money
    {
        return $this->amountMoney;
    }

    /**
     * Sets Amount Money.
     *
     * Represents an amount of money. `Money` fields can be signed or unsigned.
     * Fields that do not explicitly define whether they are signed or unsigned are
     * considered unsigned and can only hold positive amounts. For signed fields, the
     * sign of the value indicates the purpose of the money transfer. See
     * [Working with Monetary Amounts](https://developer.squareup.com/docs/build-basics/working-with-
     * monetary-amounts)
     * for more information.
     *
     * @required
     * @maps amount_money
     */
    public function setAmountMoney(Money $amountMoney): void
    {
        $this->amountMoney = $amountMoney;
    }

    /**
     * Returns Receivable Id.
     *
     * The unique ID for this [AdditionalRecipientReceivable](#type-additionalrecipientreceivable),
     * assigned by the server.
     */
    public function getReceivableId(): ?string
    {
        return $this->receivableId;
    }

    /**
     * Sets Receivable Id.
     *
     * The unique ID for this [AdditionalRecipientReceivable](#type-additionalrecipientreceivable),
     * assigned by the server.
     *
     * @maps receivable_id
     */
    public function setReceivableId(?string $receivableId): void
    {
        $this->receivableId = $receivableId;
    }

    /**
     * Encode this object to JSON
     *
     * @return mixed
     */
    public function jsonSerialize()
    {
        $json = [];
        $json['location_id']  = $this->locationId;
        $json['description']  = $this->description;
        $json['amount_money'] = $this->amountMoney;
        $json['receivable_id'] = $this->receivableId;

        return array_filter($json, function ($val) {
            return $val !== null;
        });
    }
}
