import { storeName } from './store';
import { dispatch, select } from '@wordpress/data';
import { Panel, WaitlistActiveNotice, RSVPNoticeContainer } from './settings/panel';

/**
 * Injects the waitlist controls in the tickets block.
 *
 * @since 6.2.0
 *
 * @param {Array} controls The existing controls.
 *
 * @return {Array} The updated controls.
 */
export const injectWaitlistControlsInTickets = (controls) => {
	controls.push( <Panel key="tickets" type="tickets" /> );

	return controls;
};

/**
 * Injects the waitlist controls in the rsvp block.
 *
 * @since 6.2.0
 *
 * @param {Array} controls The existing controls.
 *
 * @return {Array} The updated controls.
 */
export const injectWaitlistControlsInRSVP = (controls) => {
	controls.push( <Panel key="rsvp" type="rsvp" /> );

	return controls;
};

/**
 * Updates the store with a new ticket.
 *
 * @since 6.2.0
 *
 * @param {string} clientId      The client ID of the ticket block.
 * @param {int} ticketId      The ticket ID.
 * @param {Object} ticketDetails The ticket details.
 */
export const updateStoreWithTicket = (clientId, ticketId, ticketDetails) => {
	dispatch(storeName).reflectTicketChange( ticketId, ticketDetails, true );
};

/**
 * Updates the store with an existing ticket.
 *
 * @since 6.2.0
 *
 * @param {string} clientId      The client ID of the ticket block.
 * @param {int} ticketId      The ticket ID.
 * @param {Object} ticketDetails The ticket details.
 */
export const updateStoresTicket = (clientId, ticketId, ticketDetails) => {
	dispatch(storeName).reflectTicketChange( ticketId, ticketDetails );
};

/**
 * Updates the store with a ticket deletion.
 *
 * @since 6.2.0
 *
 * @param {string} clientId The client ID of the ticket block.
 * @param {int} ticketId The ticket ID.
 */
export const updateStoreWithoutTicket = (clientId, ticketId) => {
	dispatch(storeName).reflectTicketDeletion( ticketId );
};

/**
 * Updates the store about a RSVP upsert.
 *
 * @since 6.2.0
 *
 * @param {Object} payload The RSVP payload.
 * @param {boolean} isCreate Whether it's a create or update
 */
export const updateStoresRSVP = (payload, isCreate) => {
	dispatch(storeName).reflectRSVPChange( payload, isCreate );
};

/**
 * Removed a deleted RSVP from the store.
 *
 * @since 6.2.0
 *
 * @param {int} rsvpId The RSVP id.
 */
export const updateStoreWithoutRSVP = (rsvpId) => {
	dispatch(storeName).reflectRSVPDeletion( rsvpId );
};

/**
 * Updates the store with the assigned seating status.
 *
 * @since 6.2.0
 *
 * @param {boolean} isUsingAssignedSeating Whether the event is using assigned seating
 */
export const updateStoreWithASCStatus = (isUsingAssignedSeating) => {
	dispatch(storeName).setUsingAssignedSeating( isUsingAssignedSeating );
};

/**
 * Injects a notice to the Tickets Block when the waitlist is active.
 *
 * @since 6.2.0
 *
 * @param {Array} components The existing components.
 *
 * @return {Array} The updated components.
 */
export const injectWaitlistIsActiveNoticeForTickets = ( components ) => {
	const isActive = select( storeName ).isWaitlistActive( 'tickets' );

	if ( ! isActive ) {
		return components;
	}

	components.push( <WaitlistActiveNotice type="tickets" /> );

	return components;
};

/**
 * Injects a notice to the RSVP Block when the waitlist is active.
 *
 * @since 6.2.0
 *
 * @param {Array} components The existing components.
 *
 * @return {Array} The updated components.
 */
export const injectWaitlistIsActiveNoticeForRSVP = ( components ) => {
	const isActive = select( storeName ).isWaitlistActive( 'rsvp' );

	if ( ! isActive ) {
		components.push( <RSVPNoticeContainer /> );
		return components;
	}

	components.push( <RSVPNoticeContainer><WaitlistActiveNotice type="rsvp" /></RSVPNoticeContainer> );

	return components;
};
